%% requires MATLAB signal toolbox

% load('modelSignals.mat');                 % load signal set
load('animalSignals.mat');

[ntr,nt]=size(signals);

trainingSet=signals(1:floor(ntr/2),:);      % set used to train AR model
testSet=signals(floor(ntr/2)+1:ntr,:);      % set used to test AR model

% function to get phase of a signal (make sure it applies the correct frequency band and dt)
getPhaseMethod=@hilbertPhase;

% support functions
circ_std = @(alpha) sqrt(2*(1-abs(mean(exp(1i*alpha)))));
angdiff = @(alpha,beta) mod(alpha-beta+pi, 2*pi) - pi;

%% get AR model
ARcoefs=getARmodel(trainingSet);

%% test AR model
clear offlinePhas realtimePhas
for trN=1:size(testSet,1)
    signal=testSet(trN,:);
    sigPha=getPhaseMethod(signal);    % offline phase aquired from whole signal
    
    % real-time simulation
    truePha=zeros(1,nt);
    realtimePha=zeros(1,nt);
    tInds=floor(nt/2):nt;           % start in the middle, to give the method enough preliminary signal to use
    for t=tInds                     % get realtime signal phase at each t
        truePha(t)=sigPha(t);       % phase aquired from whole signal (to compare realtime to)
        sigSnip=signal(1:t);        % signal snippet ending at nt/2
        [pha,predSig]=realtimePhase(sigSnip,ARcoefs,getPhaseMethod);
        realtimePha(t)=pha;
    end
    
    offlinePhas(trN,:)=truePha(tInds);
    realtimePhas(trN,:)=realtimePha(tInds);
end

phaError=angdiff(offlinePhas,realtimePhas);
errorStd=circ_std(phaError(:));


%% plot test error summary
ticklabel={'-\pi','-\pi','-\pi','-\pi','0','\pi','\pi','\pi','\pi'};

h=figure();
set(h,'position',[10 200 500 800]);

subplot(2,1,1);
plot(phaError(:)/pi,offlinePhas(:)/pi,'.');
axis([-1 1 -1 1]);
set(gca,'xtick',-1:.25:1,'xticklabel',ticklabel,'ytick',-1:.25:1,'yticklabel',ticklabel);
ylabel('offline phase (rads)');
xlabel('offline-realtime difference (rads)');
title('phase error vs offline phase');
grid on;

subplot(2,1,2);
hist(phaError(:)/pi,-.995:.01:.995);
axis([-1 1 0 inf]);
set(gca,'xtick',-1:.25:1,'xticklabel',ticklabel);
xlabel('offline-realtime difference (rads)');
title({'offline vs realtime difference';['std=',num2str(errorStd/pi),'\pi']});
grid on;
